/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Data;
using System.Globalization;

using Borland.Eco.Persistence;
using Borland.Eco.Persistence.Default;
using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Persistence.Connection;

namespace Borland.Eco.Persistence.Interbase
{
	public sealed class StringAsNCharUnicode: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public string ColumnType(int length)
		{
			return String.Format(CultureInfo.InvariantCulture, "CHAR({0:d}) CHARACTER SET UNICODE_FSS", length); // do not localize
		}
		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.String));
			parameter.DbType = DbType.StringFixedLength;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(value, parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class CharAsNCharUnicode: AbstractStringSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public string ColumnType(int length)
		{
			return "NCHAR(1) CHARACTER SET UNICODE_FSS"; // do not localize
		}
		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Char));
			parameter.DbType = DbType.StringFixedLength;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = new string((System.Char)value, 1);
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToChar(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.String));
			if (DBNull.Value.Equals(columnValue) || columnValue == null)
				return null;

			char c = ((string)columnValue)[0];
			return c;
		}
	}

	public sealed class DecimalAsDecimal: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			// DECIMAL wants (precision, scale)
			// where precision is number of digits,
			// and scale is number of decimals.
			// Interbase Max size is 18
			return String.Format(CultureInfo.InvariantCulture, "DECIMAL({0:d},2)", Math.Min(length, 18)); // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Decimal));
			parameter.DbType = DbType.Decimal;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDecimal(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));			
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class Int64AsLongInteger: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMERIC(18,0)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Int64));
			parameter.DbType = DbType.Int64;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt64(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.Int64)(System.Decimal)columnValue;
		}
	}
	
	///<summary>
	///Base class mapper between an AutoInc value and its persistent representation.
	///This class must be inherited.
	///</summary>
	public abstract class AbstractAutoInc: AbstractNumericSingleColumnAttribute
	{
		///<summary>this method should never be called. Autoinc values are never stored!</summary>
		///<exception cref="NotSupportedException">Thrown always.</exception>
		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			throw new NotSupportedException(PersistenceStringRes.sDoNotCallInt32AsAutoIncValueToParameter);
		}

		///<summary>This method should never be called. Autoinc values are never stored!</summary>
		///<exception cref="NotSupportedException">Thrown always.</exception>
		public void StringToParameter(string value, IDataParameter parameter)
		{
			throw new NotSupportedException(PersistenceStringRes.sDoNotCallInt32AsAutoIncStringToParameter);
		}
		///<exception cref="ArgumentNullException">Thrown if <paramref name="database"/> is null</exception>
		public void MakeColumnAutoInc(string tableName, string columnName, IDatabase database)
		{
			if (database == null) throw new ArgumentNullException("database"); // do not localize
			string generatorName = tableName + '_' + columnName;
			generatorName = generatorName.ToUpper(CultureInfo.InvariantCulture); // interbase stores the name as uppercase...
			IExecQuery q = database.GetExecQuery();
			IQuery q2 = database.GetQuery();
			try
			{
				try
				{
					q2.AssignSqlText(System.String.Format(CultureInfo.InvariantCulture,
						"SELECT * FROM RDB$GENERATORS WHERE RDB$GENERATOR_NAME = '{0}'", // do not localize
						generatorName));
					q2.Open();
					if (q2.Eof)
					{
						q.AssignSqlText(System.String.Format(CultureInfo.InvariantCulture,
							"CREATE GENERATOR {0}", // do not localize
							generatorName));
						q.ExecSql();

						q.AssignSqlText(System.String.Format(CultureInfo.InvariantCulture,
							"SET GENERATOR {0} TO 0", // do not localize
							generatorName));
						q.ExecSql();
					}
					q2.Close();
				}
				catch
				{
					// generator names must be unique, and until we have found a
					// way to drop the generator when we drop the table, we'll
					// just ignore this exception
				}

				q.AssignSqlText(System.String.Format(CultureInfo.InvariantCulture,
					"CREATE TRIGGER AI_{2} FOR {0} BEFORE INSERT AS BEGIN NEW.{1} = GEN_ID({2}, 1); END", // do not localize
					tableName, columnName, generatorName ));

				q.ExecSql();
			}
			finally
			{
				database.ReleaseExecQuery(q);
				database.ReleaseQuery(q2);
			}
		}
	}
	
	public sealed class Int32AsAutoInc: AbstractAutoInc, IAutoIncAttributemapping
	{
		public String ColumnType(int length)
		{
			return "INTEGER"; // do not localize
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int32));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}
	
	public sealed class Int16AsAutoInc: AbstractAutoInc, IAutoIncAttributemapping
	{
		public String ColumnType(int length)
		{
			return "SMALLINT"; // do not localize
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int16));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}
}

/*
Interbase Type			BdpType			BdpSubType			System.Type

CHAR					String			stFixed			String
VARCHAR					String						String
SMALLINT				Int16						Int16
INTEGER					Int32						Int32
FLOAT					Float							Single
DOUBLE					Double							Double
BLOB Sub_Type 0			Blob			stBinary		Byte[]
BLOB Sub_Type 1			Blob			stMemo			Char[]
TIMESTAMP				Datetime						DateTime
	*/


/* Sql to create an autoinc field in Interbase

CREATE TRIGGER autoinc FOR <tablename> BEFORE INSERT AS BEGIN NEW.<autoinc_field_name> = GEN_ID(<generator_name>, ); END
CREATE GENERATOR <generator_name>; SET GENERATOR <generator_name> TO <x>;

 */
